import { useRouter } from "next/navigation";
import { packCard } from "./Packs";

interface JourneyModalProps {
  open: boolean;
  setOpen: (open: boolean) => void;
  data: JourneyData;
  packCards: packCard[];
}
type JourneyData = {
  title: string;
  totalCards: number;
  clientSeed: string;
  serverSeed: string;
  nonce: string;
  date: string;
  cards: string[];
};

export const JourneyModal = ({ open, setOpen, data, packCards }: JourneyModalProps) => {
  const router = useRouter();
  if (!open) return null;

  const formatJourneyDate = (dateStr: string) => {
    const date = new Date(dateStr.replace(" ", "T")); // safe parse

    return date
      .toLocaleDateString("en-US", {
        weekday: "long",
        month: "short",
        day: "2-digit",
        year: "numeric",
      })
      .toUpperCase();
  };

  const safeCards =
    data.cards && data.cards.length > 0
      ? data.cards
      : Array(30).fill("/images/profile/pkimg.svg");

  return (
    <div className="fixed inset-0 z-[9999] flex items-center justify-center bg-black/70 backdrop-blur-sm p-4">
      <div className="w-full max-w-6xl overflow-auto max-h-[90vh] bg-[#0c141f] text-white border border-slate-700 flex flex-col ">
        {/* HEADER */}
        <div className="relative px-8 pt-2 pr-0 pb-3 flex items-center justify-between">
          <p className="uppercase font-onest text-lg md:text-xl tracking-wide">
            <span className="font-bold">{data.title}</span>
          </p>

          <button
            onClick={() => setOpen(false)}
            className="text-slate-200 hover:text-white transition-colors"
          >
            <img
              src="/images/profile/cross.svg"
              className="p-5 pr-8 cursor-pointer"
              alt="close"
            />
          </button>
        </div>

        {/* BODY */}
        <div className="flex-1 grid grid-cols-1 md:grid-cols-2 gap-0">
          {/* LEFT: CARDS AREA */}
          <div className="relative p-6 bg-[#101925] mx-8 mb-8 flex flex-col overflow-hidden">
            {/* TOTAL CARDS badge (floating, like screenshot) */}
            <div className="flex justify-center pb-4">
              <div className="bg-black px-4 py-2 text-center shadow-lg w-fit">
                <p className="text-xs tracking-wide">TOTAL CARDS</p>
                <p className="text-2xl font-bold italic mt-1">
                  {packCards.length}
                </p>
              </div>
            </div>

            {/* inner padding so border area matches screenshot */}
            <div className=" h-[200px] overflow-y-auto no-scrollbar">
              <div className=" overflow-y-auto flex-1 grid grid-cols-5 md:grid-cols-6 lg:grid-cols-8 gap-2 md:gap-3">
                {packCards.map((card, i) => (
                  <div
                    key={i}
                    className="aspect-[2/3] relative overflow-hidden cursor-pointer"
                  >
                    <img
                      src={`/cards/${card.image}`}
                      alt={`card-${i}`}
                      className="absolute inset-0 w-full h-full object-cover"
                    />
                    <div className="pointer-events-none absolute inset-0 bg-gradient-to-t from-blue-400/25 to-transparent" />
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* RIGHT: SEEDS AREA */}
          <div className="font-sofia  pr-8  pb-8 flex flex-col">
            <div className="flex-1 space-y-2 ">
              {/* rows */}
              <div className="bg-[#101925] px-6 py-5 flex items-center justify-between">
                <span className="uppercase text-xs md:text-xl 2xl:text-2xl font-bold tracking-wide">
                  CLIENT SEED
                </span>
                <span className="font-sofia text-[11px] truncate text-xs md:text-xl 2xl:text-2xl max-w-[260px] text-right">
                  {data.clientSeed}
                </span>
              </div>
              <div className=" bg-[#101925] px-6 py-5 flex items-center justify-between">
                <span className="uppercase text-xs md:text-xl 2xl:text-2xl font-bold tracking-wide">
                  SERVER SEED
                </span>
                <span className="font-sofia text-[11px] truncate text-xs md:text-xl 2xl:text-2xl max-w-[260px] text-right">
                  {data.serverSeed}
                </span>
              </div>
              <div className="bg-[#101925] px-6 py-5 flex items-center justify-between bg-black/40">
                <span className="uppercase text-xs md:text-xl 2xl:text-2xl font-bold tracking-wide">
                  NONCE
                </span>
                <span className="font-sofia text-[11px] truncate text-xs md:text-xl 2xl:text-2xl max-w-[260px] text-right">
                  {data.nonce}
                </span>
              </div>
              {/* <div className="flex justify-center w-full"> */}
              <button
                onClick={() => {
                  router.push(
                    `/fairness?serverSeed=${encodeURIComponent(
                      data.serverSeed,
                    )}&clientSeed=${encodeURIComponent(
                      data.clientSeed,
                    )}&nonce=${data.nonce}`,
                  );
                }}
                className="bg-(--color-blue) cursor-pointer px-5 py-2 mx-auto hover:text-white transition-colors"
              >
                Go to fairness page
              </button>
              {/* </div> */}
            </div>
          </div>
        </div>

        {/* FOOTER / DATE */}
        <div className="bg-[#101925] py-4 flex justify-center items-center">
          <p className="font-sofia font-semibold text-sm md:text-base tracking-wide">
            {formatJourneyDate(data.date)}
          </p>
        </div>
      </div>
    </div>
  );
};
