import type { PackUI } from "./types";
import ChaseSlabsGrid from "./ChaseSlabsGrid";
import PackTabs from "./PackTabs";
import ValueRow from "./ValueRow";
import { MysteryPack, SlabAnalytics } from "@/app/games/mystery-packs/Inner";
import { useSelector } from "react-redux";
import { User } from "@/constants/types";

const ranges = [
  { key: "5_15", label: "5-15" },
  { key: "15_30", label: "15-30" },
  { key: "30_100", label: "30-100" },
  { key: "100_250", label: "100-250" },
  { key: "250_300", label: "250-300" },
  { key: "300_450", label: "300-450" },
] as const;

function PackDetailLayout({
  packs,
  selectedPack,
  isBuying,
  onSelect,
  onBack,
  onBuy,
  getSlabAnalytics,
  slabAnalytics,
  slabLoading,
  isCardBuying,
}: {
  packs: PackUI[];
  selectedPack: PackUI;
  isBuying: boolean;
  onSelect: (id: number) => void; // ✅ your id is number in Inner
  onBack: () => void;
  onBuy: (pack: MysteryPack) => void;
  getSlabAnalytics: (amount: any) => void;
  slabAnalytics: SlabAnalytics;
  slabLoading: boolean;
  isCardBuying: boolean;
}) {
  const sortedRanges = ranges
    .map((r) => ({
      ...r,
      percent: Number(slabAnalytics?.[r.key] ?? 0),
    }))
    .sort((a, b) => b.percent - a.percent);

  const user = useSelector((state: any) => state.auth.user) as User | null;

  return (
    <section className="w-full">
      <div className="grid gap-4 mt-6 lg:grid-cols-[1.25fr_.75fr] items-start">
        <div className=" overflow-hidden">
          <div className="bg-[#13151A] border-2 border-[#29313D] bg-img">
            <div className="relative min-h-65 md:min-h-105 flex items-center justify-center">
              <div className="absolute inset-0 " />
              <img
                src={selectedPack.image}
                alt={selectedPack.title}
                className="relative z-10 h-52 md:h-80 mt-10 lg:h-105 w-auto object-contain"
              />
            </div>

            <div className="p-3 md:p-6 bg-[#141d2b] text-white -mt-10 relative bg-blur z-99999 flex items-center justify-between gap-4">
              <div>
                <p className="font-extrabold uppercase tracking-wide text-base md:text-xl">
                  {selectedPack.title}
                </p>
                <p className="text-sm md:text-base flex items-center gap-2 opacity-80">
                  {!user ? (
                    <img
                      src="/images/header/win.svg"
                      alt=""
                      className="h-4 w-4"
                    />
                  ) : (
                    <>
                      {user?.activeMode === "coins" ? (
                        <img
                          src="/images/header/win.svg"
                          alt=""
                          className="h-4 w-4"
                        />
                      ) : (
                        <img
                          src="/images/header/star.svg"
                          alt=""
                          className="h-4 w-4"
                        />
                      )}
                    </>
                  )}
                  {selectedPack.price}
                </p>
              </div>

              <button
                disabled={isCardBuying}
                onClick={() => onBuy(selectedPack)}
                className="px-3 md:px-8 py-2 2xl:py-3 text-xs md:text-sm uppercase tracking-wide bg-(--color-blue) hover:opacity-90 disabled:opacity-60 font-onest"
              >
                {isCardBuying ? "Processing..." : "Buy Now"}
              </button>
            </div>
          </div>

          <div className="mt-5 bg-[#0f1622] border border-[#151C29] p-4 md:p-6">
            <p className="text-white font-extrabold uppercase tracking-wide text-sm mb-4">
              {selectedPack.title.split(" MYSTERY")[0]} PACK - CHASE SLABS
            </p>

            <ChaseSlabsGrid
              slabAnalytics={slabAnalytics}
              slabs={selectedPack.chaseSlabs}
              slabLoading={slabLoading}
            />
          </div>
        </div>

        <div className="bg-[#0f1622] border sticky top-24 self-start border-[#29313D] p-4 md:p-5">
          <PackTabs
            packs={packs}
            activeId={selectedPack.id}
            onSelect={onSelect}
            getSlabAnalytics={getSlabAnalytics}
          />

          <div className="mt-4">
            <p className="text-center text-white font-extrabold italic uppercase tracking-wide text-sm md:text-lg">
              {selectedPack.tier} Pack Values*
            </p>
            <p className="text-center font-onest text-xs text-[#54657E] mt-1">
              Average card value is {selectedPack.averageValue} WINCOINS
            </p>

            <div className="mt-4 space-y-3">
              {slabLoading ? (
                <div className="flex w-full h-20 items-center justify-center">
                  <div className="w-8 h-8 border-4 border-white/20 border-t-white rounded-full animate-spin" />
                </div>
              ) : (
                <>
                  {sortedRanges.map((r) => (
                    <ValueRow
                      key={r.key}
                      label={r.label}
                      percent={r.percent}
                      slabAnalytics={slabAnalytics}
                      slabLoading={slabLoading}
                    />
                  ))}
                </>
              )}
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}

export default PackDetailLayout;
