const ranges = [
    { minRange: 1, maxRange: 8, tag: "T , Foil" },
    { minRange: 1, maxRange: 54, tag: "U , Foil" },
    { minRange: 1, maxRange: 54, tag: "U , Foil" },
    { minRange: 1, maxRange: 54, tag: "U , Foil" },
    { minRange: 1, maxRange: 9, tag: "U Showcase , Foil" },
    { minRange: 1, maxRange: 49, tag: "U Etched , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot11" },
    { minRange: 1, maxRange: 10, tag: "L , Foil" },
    { minRange: 1, maxRange: 5, tag: "R Scene , Foil" },
    { minRange: 1, maxRange: 2, tag: "M Scene , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot12" },
    { minRange: 1, maxRange: 32, tag: "R , Foil" },
    { minRange: 1, maxRange: 14, tag: "M , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot13" },
    { minRange: 1, maxRange: 16, tag: "R Extended Art , Normal" },
    { minRange: 1, maxRange: 2, tag: "M Extended Art , Normal" },
    { minRange: 1, maxRange: 100, tag: "Slot14" },
    { minRange: 1, maxRange: 16, tag: "R Extended Art , Foil" },
    { minRange: 1, maxRange: 2, tag: "M Extended Art , Foil" },
    { minRange: 1, maxRange: 14, tag: "R Showcase , Foil" },
    { minRange: 1, maxRange: 5, tag: "M Showcase , Foil" },
    { minRange: 1, maxRange: 2, tag: "R Borderless , Foil" },
    { minRange: 1, maxRange: 8, tag: "M Borderless , Foil" },
    { minRange: 1, maxRange: 5, tag: "Textured , Foil" },
    { minRange: 1, maxRange: 769, tag: "Serial" },
    { minRange: 1, maxRange: 4, tag: "Serial , Foil" },
    { minRange: 1, maxRange: 100, tag: "Slot15" },
    { minRange: 1, maxRange: 32, tag: "R Etched , Foil" },
    { minRange: 1, maxRange: 14, tag: "M Etched , Foil" },
];

function incrementNonce(nonce, offset) {
  let seconds = parseInt(nonce.slice(0, 2), 10);
  let minutes = parseInt(nonce.slice(2, 4), 10);
  let hours = parseInt(nonce.slice(4, 6), 10);
  let day = parseInt(nonce.slice(6, 8), 10);
  let month = parseInt(nonce.slice(8, 10), 10);
  let year = parseInt(nonce.slice(10, 14), 10);

  seconds += offset;
  while (seconds > 59) {
    seconds -= 60;
    minutes += 1;
    if (minutes > 59) {
      minutes = 0;
      hours += 1;
      if (hours > 23) {
        hours = 0;
        day += 1;
        const daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        if (day > daysInMonth[month - 1]) {
          day = 1;
          month += 1;
          if (month > 12) {
            month = 1;
            year += 1;
          }
        }
      }
    }
  }

  return (
    seconds.toString().padStart(2, '0') +
    minutes.toString().padStart(2, '0') +
    hours.toString().padStart(2, '0') +
    day.toString().padStart(2, '0') +
    month.toString().padStart(2, '0') +
    year.toString().padStart(4, '0')
  );
}

async function generateHmacSha512(key, message) {
  const keyBuffer = new TextEncoder().encode(key);
  const msgBuffer = new TextEncoder().encode(message);
  const cryptoKey = await crypto.subtle.importKey(
    "raw",
    keyBuffer,
    { name: "HMAC", hash: "SHA-512" },
    false,
    ["sign"]
  );
  const signature = await crypto.subtle.sign("HMAC", cryptoKey, msgBuffer);
  return new Uint8Array(signature);
}

async function generateRandomNumbers(ranges, serverSeed, clientSeed, nonce) {
  const randomNumbers = [];
  const numbersPerDigest = 16;
  const digestsNeeded = Math.ceil(ranges.length / numbersPerDigest);

  for (let digestIndex = 0; digestIndex < digestsNeeded; digestIndex++) {
    const message = `${clientSeed}:${nonce}:${digestIndex}`;
    const digest = await generateHmacSha512(serverSeed, message);
    let offset = 0;

    for (let i = 0; i < numbersPerDigest && randomNumbers.length < ranges.length; i++) {
      const range = ranges[randomNumbers.length];
      const bytes = digest.slice(offset, offset + 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const num = view.getUint32(0, false);
      const rand = Math.floor((num / 0x100000000) * (range.maxRange - range.minRange + 1)) + range.minRange;
      randomNumbers.push(rand);
      offset += 4;
    }
  }

  return randomNumbers;
}

async function mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce) {
  const seenValues = {};
  const taggedResults = [];

  for (let index = 0; index < randomNumbers.length; index++) {
    if (!ranges[index]) {
      console.error(`No range found for index ${index}`);
      continue;
    }

    const { tag = "UNKNOWN", minRange, maxRange } = ranges[index];
    let finalNumber = randomNumbers[index];
    let nonceOffset = 0;

    if (!seenValues[tag]) {
      seenValues[tag] = new Set();
    }

    while (seenValues[tag].has(finalNumber.toString())) {
      nonceOffset++;
      const newNonce = incrementNonce(nonce, nonceOffset);
      const message = `${clientSeed}:${newNonce}`;
      const digest = await generateHmacSha512(serverSeed, message);
      const bytes = digest.slice(0, 4);
      const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
      const newNum = view.getUint32(0, false);
      finalNumber = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
    }

    seenValues[tag].add(finalNumber.toString());
    taggedResults.push({ value: finalNumber.toString(), tag });
  }

  return taggedResults;
}

async function filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce) {
    let filteredResults = [];
    let nonceOffset = 0;
    const usedIndexes = new Set();

    const findAndConsume = (tag) => {
        const item = taggedResults.find((item, i) => item.tag === tag && !usedIndexes.has(i));
        if (item) usedIndexes.add(taggedResults.indexOf(item));
        return item;
    };

    const add = (item) => { if (item) filteredResults.push(item); };

    let tFrameFoil = taggedResults.filter((item, i) => item.tag === "T , Foil" && !usedIndexes.has(i));
    tFrameFoil.slice(0, 1).forEach(item => { usedIndexes.add(taggedResults.indexOf(item)); filteredResults.push(item); });

    let uFoil = taggedResults.filter((item, i) => item.tag === "U , Foil" && !usedIndexes.has(i));
    uFoil.slice(0, 3).forEach(item => { usedIndexes.add(taggedResults.indexOf(item)); filteredResults.push(item); });

    let uShow = taggedResults.filter((item, i) => item.tag === "U Showcase , Foil" && !usedIndexes.has(i));
    uShow.slice(0, 1).forEach(item => { usedIndexes.add(taggedResults.indexOf(item)); filteredResults.push(item); });

    let uEtch = taggedResults.filter((item, i) => item.tag === "U Etched , Foil" && !usedIndexes.has(i));
    uEtch.slice(0, 1).forEach(item => { usedIndexes.add(taggedResults.indexOf(item)); filteredResults.push(item); });

    const slot11 = taggedResults.find(item => item.tag === "Slot11");
    let slot11Item = null;
    let slot11Tag = null;
    if (slot11) {
        const num = parseInt(slot11.value, 10);
        if (num <= 87) {
            slot11Tag = "L , Foil";
            slot11Item = findAndConsume("L , Foil");
        } else if (num <= 99) {
            slot11Tag = "R Scene , Foil";
            slot11Item = findAndConsume("R Scene , Foil");
        } else {
            slot11Tag = "M Scene , Foil";
            slot11Item = findAndConsume("M Scene , Foil");
        }
        add(slot11Item);
    }

    const slot12 = taggedResults.find(item => item.tag === "Slot12");
    let slot12Item = null;
    let slot12Tag = null;
    if (slot12) {
        const num = parseInt(slot12.value, 10);
        if (num <= 82) {
            slot12Tag = "R , Foil";
            slot12Item = findAndConsume("R , Foil");
        } else {
            slot12Tag = "M , Foil";
            slot12Item = findAndConsume("M , Foil");
        }

        if (slot11Item && slot12Item && slot11Tag === slot12Tag && slot11Item.value === slot12Item.value) {
            const rangeForTag = ranges.find(range => range.tag === slot12Tag);
            if (rangeForTag) {
                const { minRange, maxRange } = rangeForTag;
                let newValue;
                do {
                    nonceOffset++;
                    const newNonce = incrementNonce(nonce, nonceOffset);
                    const message = `${clientSeed}:${newNonce}`;
                    const digest = await generateHmacSha512(serverSeed, message);
                    const bytes = digest.slice(0, 4);
                    const view = new DataView(bytes.buffer, bytes.byteOffset, 4);
                    const newNum = view.getUint32(0, false);
                    newValue = Math.floor((newNum / 0x100000000) * (maxRange - minRange + 1)) + minRange;
                } while (newValue.toString() === slot11Item.value);

                slot12Item = { value: newValue.toString(), tag: slot12Tag };
            }
        }
        add(slot12Item);
    }

    const slot13 = taggedResults.find(item => item.tag === "Slot13");
    if (slot13) {
        const num = parseInt(slot13.value, 10);
        if (num <= 97) {
            add(findAndConsume("R Extended Art , Normal"));
        } else {
            add(findAndConsume("M Extended Art , Normal"));
        }
    }

    const serial = taggedResults.find(item => item.tag === "Serial");
    if (serial) {
        if (serial.value === "769") {
            const serialFoil = findAndConsume("Serial , Foil");
            if (serialFoil) {
                add(serialFoil);
            }
        } else {
            add(serial);
            usedIndexes.add(taggedResults.indexOf(serial));
            const slot14 = taggedResults.find(item => item.tag === "Slot14");
            if (slot14) {
                const num = parseInt(slot14.value, 10);
                if (num <= 40) add(findAndConsume("R Extended Art , Foil"));
                else if (num <= 41) add(findAndConsume("M Extended Art , Foil"));
                else if (num <= 76) add(findAndConsume("R Showcase , Foil"));
                else if (num <= 82) add(findAndConsume("M Showcase , Foil"));
                else if (num <= 92) add(findAndConsume("R Borderless , Foil"));
                else if (num <= 97) add(findAndConsume("M Borderless , Foil"));
                else if (num <= 100) add(findAndConsume("Textured , Foil"));
            }
        }
    }

    const slot15 = taggedResults.find(item => item.tag === "Slot15");
    if (slot15) {
        const num = parseInt(slot15.value, 10);
        if (num <= 82) {
            add(findAndConsume("R Etched , Foil"));
        } else {
            add(findAndConsume("M Etched , Foil"));
        }
    }

    return filteredResults;
}

async function setData(serverSeed, clientSeed, nonce) {
  const randomNumbers = await generateRandomNumbers(ranges, serverSeed, clientSeed, nonce);
  const taggedResults = await mapTaggedResults(randomNumbers, ranges, serverSeed, clientSeed, nonce);
  const filteredResults = await filterResults(taggedResults, ranges, serverSeed, clientSeed, nonce);

  return {
    success: true,
    result: filteredResults,
    nonce,
    serverSeed,
    clientSeed
  };
}
export { setData, ranges};
